%{
Calculates a "phase diagram" for a material mixture, in this case for a
mixture of a martensitic stainless steel with Alloy 800.

This type of calculation is for example useful for understanding effects
when welding dissimilar materials - without the need to perform diffusion
calculations.
%}
import tc_toolbox.material_to_material.*;

session = tc_toolbox.TCToolbox();

independent_elements = ["Cr", "Ni"];
material_a_composition = [17.0, 2.0];
material_b_composition = [19.0, 35.0];

[filepath,name,ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder( name + "_cache");

sb = session.select_database_and_elements("FEDEMO", ["Fe", "Cr", "Ni"]);
tc_system = sb.get_system();

calc = tc_system.with_material_to_material().with_phase_diagram_calculation();
calc.set_material_a(containers.Map(independent_elements, material_a_composition), "Fe");
calc.set_material_b(containers.Map(independent_elements, material_b_composition), "Fe");
calc.set_composition_unit(tc_toolbox.CompositionUnit.MASS_PERCENT);
calc.with_first_axis(MaterialToMaterialCalculationAxis.fraction_of_material_b());
calc.with_second_axis(MaterialToMaterialCalculationAxis.temperature(300 + 273.15, 800 + 273.15, 600 + 273.15));
calculation_results = calc.calculate();

calculation_results...
    .add_coordinate_for_phase_label(0.5, 770 + 273.15)...
    .add_coordinate_for_phase_label(0.58, 600 + 273.15)...
    .add_coordinate_for_phase_label(0.2, 530 + 273.15)...
    .add_coordinate_for_phase_label(0.37, 410 + 273.15)...
    .add_coordinate_for_phase_label(0.335, 310 + 273.15)...
    .add_coordinate_for_phase_label(0.05, 740 + 273.15)...
    .add_coordinate_for_phase_label(0.01, 360 + 273.15)...
    .add_coordinate_for_phase_label(0.01, 560 + 273.15)...
    .add_coordinate_for_phase_label(0.87, 440 + 273.15)...
    .add_coordinate_for_phase_label(0.03, 490 + 273.15);

phase_diagram_data = calculation_results.get_values_grouped_by_stable_phases_of(...
    tc_toolbox.Constants.MATERIAL_B_FRACTION,...
    tc_toolbox.ThermodynamicQuantity.temperature());

calculation_results.remove_phase_labels();

plot_data_compositions = calculation_results.get_values_grouped_by_quantity_of(...
    tc_toolbox.Constants.MATERIAL_B_FRACTION,...
    tc_toolbox.ThermodynamicQuantity.mass_fraction_of_a_component(...
        tc_toolbox.Constants.ALL_COMPONENTS));

figure()

subplot(2,1,1);
title("Martensitic stainless steel - Alloy 800");
plot_phase_diagram(phase_diagram_data, "Fraction of Alloy 800 [-]", "Temperature [{\circ}C]");
ylim([300, 800])

subplot(2,1,2);
plot_compositions(plot_data_compositions, "Fraction of Alloy 800 [-]", "Composition [wt-%]");


function plot_phase_diagram(phase_diagram_data, x_axis_label, y_axis_label)
    xlabel(x_axis_label);
    ylabel(y_axis_label);
    xlim([0 1]);

    hold on

    tie_lines = phase_diagram_data.get_tie_lines();
    plot(tie_lines.get_x(), tie_lines.get_y() - 273.15, "g");

    invariant = phase_diagram_data.get_invariants();
    plot(invariant.get_x(), invariant.get_y() - 273.15, "r");

    lines = phase_diagram_data.get_lines();
    plots_in_legend = zeros(length(lines), 1);
    index = 1;
    for k = lines.keys()
        line = lines(k{1});
        plots_in_legend(index) = plot(line.get_x(), line.get_y() - 273.15, ...
            "DisplayName", line.get_label(), "LineWidth", 2);
        index = index + 1;
    end

    phase_labels = phase_diagram_data.get_phase_labels();
    for i = 1:length(phase_labels)
        text_legend = text(phase_labels{i}.get_x() + 0.01, phase_labels{i}.get_y() - 273.15 + 15, ...
            phase_labels{i}.get_text());
        scatter(phase_labels{i}.get_x(), phase_labels{i}.get_y() - 273.15, 3.0, 'k', 'filled');
        set(text_legend, "Interpreter", "none");
    end

    legend(plots_in_legend);
    leg = findobj(gcf, "Type", "Legend");
    set(leg, "Interpreter", "none");
    hold off
end

function plot_compositions(plot_data, x_axis_label, y_axis_label)
    xlabel(x_axis_label);
    ylabel(y_axis_label);

    hold on
    for k = plot_data.get_lines().values()
        group = k{1};
        plot(group.get_x(), 100 * group.get_y(), "DisplayName",...
            group.get_label(), "LineWidth", 2);
    end
    hold off
    legend;
    leg = findobj(gcf, "Type", "Legend");
    set(leg, "Interpreter", "none");
end