%{
Calculates the phases occurring during a gradual transition between two
materials, in this case from a martensitic stainless steel to Alloy 800.

This type of calculation is for example useful for understanding effects
when welding dissimilar materials - without the need to perform diffusion
calculations.
%}

import tc_toolbox.material_to_material.*;

independent_elements = ["Cr", "Ni"];
material_a_composition = [17.0, 2.0];
material_b_composition = [19.0, 35.0];

session = tc_toolbox.TCToolbox();

[filepath,name,ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder( name + "_cache");

sb = session.select_database_and_elements("FEDEMO", ["Fe", "Cr", "Ni"]);
tc_system = sb.get_system();

calc = tc_system.with_material_to_material().with_property_diagram_calculation();
calc.set_material_a(containers.Map(independent_elements, material_a_composition), "Fe");
calc.set_material_b(containers.Map(independent_elements, material_b_composition), "Fe");
calc.set_composition_unit(tc_toolbox.CompositionUnit.MASS_PERCENT);
calc.with_axis(MaterialToMaterialCalculationAxis.fraction_of_material_b(0.0, 1.0, 0.5));
calc.with_constant_condition(ConstantCondition.temperature(650 + 273.15));
property_diagram = calc.calculate();

property_diagram.set_phase_name_style(...
    tc_toolbox.step_or_map_diagrams.PhaseNameStyle.ALL);
plot_data_step = property_diagram.get_values_grouped_by_quantity_of(...
    tc_toolbox.Constants.MATERIAL_B_FRACTION,...
    tc_toolbox.ThermodynamicQuantity.volume_fraction_of_a_phase(...
        tc_toolbox.Constants.ALL_PHASES));

plot_data_compositions = property_diagram.get_values_grouped_by_quantity_of(...
    tc_toolbox.Constants.MATERIAL_B_FRACTION,...
    tc_toolbox.ThermodynamicQuantity.mass_fraction_of_a_component(tc_toolbox.Constants.ALL_COMPONENTS));

figure()

subplot(2,1,1);
title("Martensitic stainless steel - Alloy 800");
plot_subplot(plot_data_step, "Fraction of Alloy 800", "Volume fraction of phases [-]");

subplot(2,1,2);
plot_subplot(plot_data_compositions, "Fraction of Alloy 800", "Mass-fraction components [-]");


function plot_subplot(plot_data, x_axis_label, y_axis_label)
    xlabel(x_axis_label);
    ylabel(y_axis_label);

    hold on
    for k = plot_data.keys()
        group = plot_data(k{1});
        plot(group.get_x(), group.get_y(), "DisplayName", group.get_label(), ...
            "LineWidth", 2);
    end
    hold off
    legend;
    leg = findobj(gcf, "Type", "Legend");
    set(leg, "Interpreter", "none");
end