%{
Calculates solidification using Scheil with backdiffusion in the primary
phase, and compares the result to other calculation types.
%}
thermodynamic_database = "ALDEMO";
kinetic_database = "MALDEMO";
dependent_element = "Al";
composition = containers.Map({'Cu',   'Si'}, ... % in mole percent
                             {0.021 , 0.01});
elements = string(composition.keys());

cooling_rate = 0.1; % in K/s
% secondary dendrite arm spacing scaling factor in c*cooling_rate^-n (in meters)
sdas_c = 5.0e-5; 
sdas_n = 0.33;

session = tc_toolbox.TCToolbox();

[filepath,name,ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder( name + "_cache");

tc_system = session.select_thermodynamic_and_kinetic_databases_with_elements(...
                      thermodynamic_database, kinetic_database, ...
                        [dependent_element elements])...
                     .get_system();


% setup and perform a regular Scheil calculation for comparison
% perform the Scheil calculation for the same system
scheil_calculator = tc_system...
                      .with_scheil_calculation()...
                      .set_composition_unit(...
                        tc_toolbox.CompositionUnit.MOLE_FRACTION);

for element = elements
    scheil_calculator.set_composition(element, composition(element));
end

result = scheil_calculator.calculate();

[fraction_solid_regular_scheil, temp_regular_scheil] = result.get_values_of(...
    tc_toolbox.ScheilQuantity.mole_fraction_of_all_solid_phases(),...
    tc_toolbox.ScheilQuantity.temperature());
                                                        

% setup the back diffusion options and perform Scheil calculation with back diffusion
scheil_back_diffusion_calculator = ...
    scheil_calculator.with_calculation_type(tc_toolbox.scheil.ScheilCalculationType.scheil_back_diffusion()...
        .calculate_secondary_dendrite_arm_spacing()...
        .set_cooling_rate(cooling_rate)...
        .set_c(sdas_c)...
        .set_n(sdas_n)...
        .set_primary_phasename("AUTOMATIC"));

result = scheil_back_diffusion_calculator.calculate();
        
[fraction_solid_back_diffusion, temp_back_diffusion] = result.get_values_of(...
    tc_toolbox.ScheilQuantity.mole_fraction_of_all_solid_phases(),...
    tc_toolbox.ScheilQuantity.temperature()...
);
                                                        
% calculate the equilbrium line using step calculation
npm_liq_eq = get_equilibrium_line(tc_system, elements, ...
    temp_back_diffusion(1), temp_back_diffusion(end), composition);

% calculate solidification line also using Dictra
[fraction_solid_dictra, temp_dictra] = ...
    calculate_solidification_curve_with_dictra(...
        tc_system, ...
        elements, ...
        composition,...
        temp_back_diffusion(1),...
        temp_back_diffusion(end),...
        cooling_rate,...
        sdas_c, sdas_n...
    );
                                                                 
                                                                            
figure();
sgtitle("Al-2.1Cu-1Si Mole%");
xlabel("Fraction solid phase");
ylabel("Temperature (K)");
hold on   
plot(fraction_solid_back_diffusion, temp_back_diffusion, "DisplayName", ...
    "Scheil with back diffusion");
plot(fraction_solid_dictra, temp_dictra, "DisplayName", "Dictra");
plot(fraction_solid_regular_scheil, temp_regular_scheil, "DisplayName", ...
    "Scheil");
for k = npm_liq_eq.keys()
    label = k{1};  
    group = npm_liq_eq(k{1});
    fraction_solid_eq = 1.0 - group.get_x;
    plot(fraction_solid_eq, group.get_y, "--", "DisplayName", "Equlibrium", ...
    "LineWidth", 2);
end    
    
legend

        
 % Calculates the equilibrium line using step calculation
 function [result] = get_equilibrium_line(tc_system, elements, liq_temp, temperature_stop, composition)
    temp_step = 1.0;
    step_calculator = ...
        tc_system.with_property_diagram_calculation()...
            .with_axis(tc_toolbox.step_or_map_diagrams.CalculationAxis("T")...
                .set_min(temperature_stop)...
                .set_max(liq_temp)...
                .with_axis_type(...
                    tc_toolbox.step_or_map_diagrams.AxisType.linear()...
                    .set_max_step_size(temp_step))...
            );

    for element = elements
        step_calculator.set_condition(...
            tc_toolbox.ThermodynamicQuantity.mole_fraction_of_a_component(element), ...
            composition(element)...
        );
    end
                         
    step_result = step_calculator.calculate();
    result = step_result.get_values_grouped_by_quantity_of("NPM(LIQ)", "T");
 end
 
 
% Calculates a solidification curve using the dictra diffusion module
function [fraction_solid_dictra, temp_dictra] = calculate_solidification_curve_with_dictra(tc_system, elements, composition, liq_temp, temperature_stop, cooling_rate, sdas_c, sdas_n)
    temp_lowest_dictra = 770.0;
    time_lowest_dictra = (liq_temp - temp_lowest_dictra) / cooling_rate;
    time_stop = (liq_temp - temperature_stop) / cooling_rate;
    sim_time_dictra = min(time_stop, time_lowest_dictra);

    liq_phase = "LIQUID";
    dictra_primary_phase = "FCC_A1";

    secondary_dendrite_arm_spacing = sdas_c * cooling_rate ^(-sdas_n);
    composition_profile = ...
        tc_toolbox.diffusion.CompositionProfile(...
            tc_toolbox.diffusion.Unit.MOLE_FRACTION);
    for element = elements
        composition_profile.add(element, ...
            tc_toolbox.diffusion.ElementProfile.constant(composition(element)));
    end

    dictra_calc = ...
        tc_system.with_non_isothermal_diffusion_calculation()...
            .with_temperature_profile(tc_toolbox.TemperatureProfile()...
                .add_time_temperature(0, liq_temp)...
                .add_time_temperature(time_stop, temperature_stop)...
            )...
            .set_simulation_time(sim_time_dictra)...
            .with_timestep_control(...
                tc_toolbox.diffusion.TimestepControl()...
                    .enable_check_interface_position())...
            .add_region(tc_toolbox.diffusion.Region("Liquid")...
                .set_width(secondary_dendrite_arm_spacing)...
                .with_grid(...
                    tc_toolbox.diffusion.CalculatedGrid.geometric()...
                        .set_no_of_points(50)...
                        .set_geometrical_factor(0.95)...
                )...
                .with_composition_profile(composition_profile)...
                .add_phase(liq_phase)...
                .add_phase_allowed_to_form_at_right_interface(...
                    dictra_primary_phase, 1e-5)...
            );
                              
    dictra_result = dictra_calc.calculate();

    [temp_dictra, poi] = dictra_result.get_values_of(...
        "T", ...
        "Position-Of-Interface Liquid upper", ...
        "interface Liquid upper"...
    );
    fraction_solid_dictra = 1.0 - poi ./ secondary_dendrite_arm_spacing;
end

