%{
Moving Boundary Diffusion Couple for an Fe-Ni-Cr Alloy

This example simulates the diffusion paths in a moving boundary problem between two multiphase regions. 
Due to the multiple phases, the default `AutomaticSolver` uses the 'homogenization mode'. Initially 
there is a concentration profile with a step between the regions.

The example uses a minimum number of required settings. Default values are used for any unspecified settings.
%}

session = tc_toolbox.TCToolbox();
[filepath, name, ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder(name + "_cache");

tc_system = session...
                .select_thermodynamic_and_kinetic_databases_with_elements("FEDEMO", "MFEDEMO", ["Fe", "Cr", "Ni"])...
                .without_default_phases()...
                .select_phase("FCC_A1")...
                .select_phase("BCC_A2")...
                .get_system();

calculator = tc_system...
                .with_isothermal_diffusion_calculation()...
                .set_temperature(1100 + 273.15)...
                .set_simulation_time(3600 * 1000)...
                .add_region(tc_toolbox.diffusion.Region("alpha")...
                                .set_width(9.345E-5)...
                                .with_grid(tc_toolbox.diffusion.CalculatedGrid.automatic())...
                                .with_composition_profile(tc_toolbox.diffusion.CompositionProfile(tc_toolbox.diffusion.Unit.MASS_FRACTION)...
                                                            .add("Cr", tc_toolbox.diffusion.ElementProfile.constant(0.38))...
                                                            .add("Ni", tc_toolbox.diffusion.ElementProfile.constant(1.0E-5)))...
                                .add_phase("BCC_A2"))...
                .add_region(tc_toolbox.diffusion.Region("gamma")...
                                .set_width(5.0E-4)...
                                .with_grid(tc_toolbox.diffusion.CalculatedGrid.automatic())...
                                .with_composition_profile(tc_toolbox.diffusion.CompositionProfile(tc_toolbox.diffusion.Unit.MASS_FRACTION)...
                                                          .add("Cr", tc_toolbox.diffusion.ElementProfile.constant(0.27))...
                                                          .add("Ni", tc_toolbox.diffusion.ElementProfile.constant(0.28)))...
                                .add_phase("FCC_A1"));

result = calculator.calculate();

% Plot result
title("Diffusion paths in a Fe-Cr-Ni diffusion couple");
xlabel("Ni-content [wt-%]");
ylabel("Cr-content [wt-%]");
xlim([0 40]);
ylim([0 40]);

hold on;
for plot_time = [1, 10, 100, 1000]
    [ni, cr] = result.get_values_of(tc_toolbox.DiffusionQuantity.mass_fraction_of_a_component("Ni"),... 
                               tc_toolbox.DiffusionQuantity.mass_fraction_of_a_component("Cr"),... 
                               tc_toolbox.PlotCondition.time(plot_time * 3600.0),.... 
                               tc_toolbox.IndependentVariable.distance());
    label = num2str(plot_time) + "h";
    plot(ni * 100, cr * 100, 'DisplayName', label);
end
hold off;
