%{
Diffusion with an Inactive Phase Forming in the Presence of a Driving Force

This example simulates a moving boundary problem that is initially a one-phase austenite region. 
Due to the available driving force, a new ferrite region is starting to grow at the left interface 
of the austenite region. 

Initially a constant concentration profile is defined within the single-phase region.

The example uses a minimum number of required settings. Default values are used for any 
%}

session = tc_toolbox.TCToolbox();
[filepath, name, ext] = fileparts(mfilename("fullpath"));
session.set_cache_folder(name + "_cache");

tc_system = session...
            .select_thermodynamic_and_kinetic_databases_with_elements("FEDEMO", "MFEDEMO", ["Fe", "C"])...
            .without_default_phases()...
            .select_phase("FCC_A1")...
            .select_phase("BCC_A2")...
            .get_system();

calculator = tc_system...
                .with_non_isothermal_diffusion_calculation()...
                .with_temperature_profile(tc_toolbox.TemperatureProfile()...
                                            .add_time_temperature(0, 1173)...
                                            .add_time_temperature(773, 1050)...
                                            .add_time_temperature(1000, 1050))...
                .set_simulation_time(1000)...
                .add_region(tc_toolbox.diffusion.Region("Austenite")...
                                .set_width(2000e-6)...
                                .with_grid(tc_toolbox.diffusion.CalculatedGrid.automatic())...
                                .with_composition_profile(tc_toolbox.diffusion.CompositionProfile(tc_toolbox.diffusion.Unit.MASS_PERCENT)...
                                                            .add("C", tc_toolbox.diffusion.ElementProfile.constant(0.15)))...
                                .add_phase("FCC_A1")...
                                .add_phase_allowed_to_form_at_left_interface("BCC_A2"));

result = calculator.calculate();

title("Mass fraction of C");
xlabel("Distance [\mum]");
ylabel("C-content [wt-%]");
xlim([0 150]);
ylim([0 0.5]);

hold on;
for plot_time = [400, 600, 800, 1000]
    [distance, c] = result.get_mass_fraction_of_component_at_time("C", plot_time);
    plot(distance * 1e6, c * 100, 'DisplayName', num2str(plot_time));
end
hold off;
