from tc_python import *

"""
These examples show how to manage licenses that are activated with user credentials in TC-Python.
"""


"""
Prints information about the currently activated license.
"""
def print_info():
   with TCPython() as start:
       license_manager = start.get_license_manager()
       print("License Information:")
       print(license_manager.get_info())


"""
Activate the license using provided user credentials over the internet.
This requires a valid username and password.
After activation, the updated license information will be displayed.
"""
def activate_online():
   with TCPython() as start:
       usr = input("User:")
       pwd = input("Password:")
       license_manager = start.get_license_manager()
       license_manager.activate(user=usr, password=pwd)
       print(license_manager.get_info())


"""
Create an offline activation file in the supplied directory. It is used to activate the license on a machine that does not have internet access.
The file is created using user credentials. This is the first step to activate an offline license.

Copy that file to a computer with internet access and go to the website https://offline.licensespring.com/. Upload the file as instructed. You will 
then get an offline activation file to download. Copy that file to the offline computer and perform activate_offline()
"""
def create_offline_activation_file():
   with TCPython() as start:
       offline_file_directory = os.path.dirname(os.path.abspath(__file__))
       license_manager = start.get_license_manager()
       usr = input("User:")
       pwd = input("Password:")
       license_manager.create_offline_activation_file(user=usr, password=pwd, path=offline_file_directory)
       print(f"Offline activation file created at {offline_file_directory}.")


"""
Activate an offline license using the offline activation file (see create_offline_activation_file()).
"""
def activate_offline():
   with TCPython() as start:
       offline_activation_file_directory = os.path.dirname(os.path.abspath(__file__))
       license_manager = start.get_license_manager()
       usr = input("User:")
       pwd = input("Password:")
       license_manager.activate_offline(user=usr, password=pwd, path=offline_activation_file_directory)
       print(start.get_license_manager().get_info())

"""
Demonstrates how to use proxy settings for the LicenseManager.
This function showcases the `proxy_settings_set` method for configuring
proxy settings when working with the license.
"""
def use_proxy():
    with TCPython() as start:
        license_manager = start.get_license_manager()

        # Define proxy settings (example values)
        proxy_host = "http://proxy.example.com"  # Proxy server URL
        proxy_port = 8080  # Proxy server port
        proxy_username = "user"  # Proxy username (if required)
        proxy_password = "password"  # Proxy password (if required)

        # Set the proxy settings
        try:
            license_manager.proxy_settings_set(
                host=proxy_host,
                port=proxy_port,
                user=proxy_username,
                password=proxy_password
            )
            print("Proxy settings configured successfully!")
        except Exception as e:
            print(f"Failed to set proxy settings: {e}")

        # Remove the proxy settings (revert to direct connection)
        # use this if you are no longer behind a proxy server.
        # license_manager.proxy_settings_remove()


"""
Update the current license to its latest state.
This is usually done to refresh the license information and ensure it is up-to-date.
Can not be performed with an offline license.
"""
def update_license():
   with TCPython() as start:
       license_manager = start.get_license_manager()
       license_manager.update()
       print(license_manager.get_info())


"""
Deactivate the currently active license.

This is typically done when you no longer need the license on this machine or want to activate it elsewhere.

If you are using an offline license, this will create an offline deactivation file, which is used to deactivate the license on a machine without internet access.
Copy that file to a computer with internet access and go to the website https://offline.licensespring.com/. 
Upload the file as instructed. The license will then be properly deactivated.

An online license will be directly deactivated with this function.
"""
def deactivate_license():
   with TCPython() as start:
       license_manager = start.get_license_manager()
       license_manager.deactivate()
       print(license_manager.get_info())


"""
Create an offline deactivation file, which is used to deactivate the license on a machine without internet access.

Copy that file to a computer with internet access and go to the website https://offline.licensespring.com/. 

Upload the file as instructed. The license will then be properly deactivated.
"""
def deactivate_offline_license():
   with TCPython() as start:
      license_manager = start.get_license_manager()
      offline_file_directory = os.path.dirname(os.path.abspath(__file__))
      license_manager.deactivate_offline(path=offline_file_directory)


# Note: If your system is behind a proxy, you must call this function and set up
#       the proxy settings before invoking other functions that require network
#       connectivity (e.g., activate_online, update_license, deactivate_license).

# use_proxy()

print_info()
# activate_online()
# create_offline_activation_file()
# activate_offline()
# deactivate_license()
# deactivate_offline_license()