from tc_python import *
import matplotlib.pyplot as plt
from matplotlib.pyplot import *

"""
In this example the thermal gradients and solidification rates are calculated after a AM steady state simulation for the alloy IN718.
The property model Columnar-to-Equiaxed-Transition (CET) is used to calculate the transition limits for same alloy and range of 
thermal gradients and solidification rates.

The results from the two calculations are overlaid in a 2d plot showing that most solidification conditions are for columnar growth. 
The thermal gradients and solidification rates are visualized in 3d plots.  
"""

def get_material_properties_from_scheil(system, solutes):
    scheil_options = (ScheilOptions().calculate_from_start_temperature()
                      .calculate_to_temperature_below_solidus()
                      .enable_evaporation_property_calculation())

    scheil_calculator = (system.with_scheil_calculation()
                         .with_options(scheil_options)
                         .set_start_temperature(5000.0)
                         .set_composition_unit(CompositionUnit.MASS_PERCENT))

    for element in solutes:
        scheil_calculator.set_composition(element, solutes[element])

    scheil_result = scheil_calculator.calculate()
    return MaterialProperties.from_scheil_result(scheil_result)


def get_CET_lines(system, solutes, primary_phase="FCC_L12", interfacial_energy=0.5, nb_nucleations_site=2e15, nucleation_undercooling=2.5, equiaxed_exponent=3.4, equaixed_fractions=[0.0066, 0.49, 0.99]):
    calc = system.with_property_model_calculation("CETPythonModel")
    calc.set_composition_unit(CompositionUnit.MASS_PERCENT)

    for element in solutes:
        calc.set_composition(element, solutes[element])

    logv_list = np.linspace(-5,0, 100)
    v = []
    thermal_gradients = {}
    calc.set_argument("primaryPhase", primary_phase)
    calc.set_argument("InterfacialEnergy", str(interfacial_energy))
    calc.set_argument("NbNucleationSites", str(nb_nucleations_site))
    calc.set_argument("NucleationUndercooling", str(nucleation_undercooling))
    calc.set_argument("EquiaxedExponent", str(equiaxed_exponent))
    calc.set_argument("Equiaxed fractions", str(equaixed_fractions).replace('[','').replace(']','').replace(',',''))

    for logv in logv_list:
        calc.set_argument("Log10(v)", logv)
        result = calc.calculate()
        thermal_gradient = result.get_value_of("Thermal gradient")
        for key, value in thermal_gradient.items():
            if key not in thermal_gradients:
                thermal_gradients[key] = []
            thermal_gradients[key].append(value)
        v.append(10**logv)
        print("CET for log(v):{:.2f}m/s".format(logv))
    return v, thermal_gradients


with TCPython() as start:
    start.set_cache_folder(os.path.basename(__file__) + "_cache")

    thermodynamic_database = "TCNI12"
    kinetic_database = "MOBNI6"
    dependent_element = "Ni"
    solutes = {"Cr": 17.5, "Fe": 18.4, "Nb": 4.8, "Mo": 3.08, "Ti": 0.878, "Al": 0.428, "C": 0.0321}
    scan_speed = 1.0 #m/s

    elements = [dependent_element] + list(solutes.keys())
    system = start.select_thermodynamic_and_kinetic_databases_with_elements(thermodynamic_database, kinetic_database,  elements).get_system()

    solidification_rate_CET, thermal_gradients_CET = get_CET_lines(system, solutes, primary_phase="FCC_L12", interfacial_energy=0.5, nb_nucleations_site=4.0E11,
                                                                   nucleation_undercooling=4.0, equiaxed_exponent=3.13, equaixed_fractions=[0.01, 0.49, 0.99])

    mp = get_material_properties_from_scheil(system, solutes)

    am_calc = (start.with_additive_manufacturing()
                 .with_steady_state_calculation()
                 .with_heat_source(HeatSource.gaussian_with_constant_absorptivity().with_keyhole_model(KeyholeModel())
                                             .set_beam_radius(45.0e-6)
                                             .set_absorptivity(36.0)
                                             .set_power(250)
                                             .set_scanning_speed(scan_speed))
                 .with_material_properties(mp)
            .enable_fluid_flow_marangoni())

    am_result = am_calc.calculate()

    thermal_gradient_AM, solidification_rate_AM, x, y, z = am_result.get_thermal_gradient_and_solidification_rate()
    thermal_gradient_AM_melting, melting_rate_AM, xm, ym, zm = am_result.get_thermal_gradient_and_melting_rate()

    fig, axs = plt.subplots(1, 1)
    axs.title.set_text("Columnar to Equiaxed Transition for IN718\n Thermal gradient and solidification rate from AM Steady-state simulation")
    for label, thermal_gradient_CET in thermal_gradients_CET.items():
        axs.loglog(solidification_rate_CET, thermal_gradient_CET, label=label)

    axs.loglog(solidification_rate_AM, thermal_gradient_AM, label='Thermal gradient and solidification rate from AM Steady-state simulation', marker='o', lw=0)
    axs.set_xlabel('Solidification rate (m/s)')
    axs.set_ylabel('Thermal Gradient (K/s) ')
    axs.legend()

    fig2 = plt.figure()
    fig2.suptitle('From AM steady-state')
    ax1 = fig2.add_subplot(121, projection='3d')
    p3d1 = ax1.scatter(x, y, z, s=30, c=solidification_rate_AM, marker='o', cmap='jet')
    ax1.title.set_text('Solidification rate(m/s)')
    ax1.set_xlabel('x')
    ax1.set_ylabel('y')
    ax1.set_zlabel('z')
    fig2.colorbar(p3d1, ax=ax1)

    ax2 = fig2.add_subplot(122, projection='3d')
    p3d2 = ax2.scatter(x, y, z, s=30, c=thermal_gradient_AM, marker='o', cmap='jet')
    ax2.title.set_text('Thermal gradient (K/m)')
    ax2.set_xlabel('x')
    ax2.set_ylabel('y')
    ax2.set_zlabel('z')
    fig2.colorbar(p3d2, ax=ax2)

    fig3 = plt.figure()
    fig3.suptitle('From AM steady-state')
    ax1 = fig3.add_subplot(121, projection='3d')
    p3d1 = ax1.scatter(xm, ym, zm, s=30, c=melting_rate_AM, marker='o', cmap='jet')
    ax1.title.set_text('Melting rate(m/s)')
    ax1.set_xlabel('x')
    ax1.set_ylabel('y')
    ax1.set_zlabel('z')
    fig3.colorbar(p3d1, ax=ax1)

    ax2 = fig3.add_subplot(122, projection='3d')
    p3d2 = ax2.scatter(xm, ym, zm, s=30, c=thermal_gradient_AM_melting, marker='o', cmap='jet')
    ax2.title.set_text('Thermal gradient (K/m)')
    ax2.set_xlabel('x')
    ax2.set_ylabel('y')
    ax2.set_zlabel('z')
    fig3.colorbar(p3d2, ax=ax2)

    plt.show()
